package com.rapidminer.krimp;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import com.rapidminer.example.Attribute;
import com.rapidminer.example.Example;
import com.rapidminer.example.ExampleSet;
import com.rapidminer.operator.learner.associations.BooleanAttributeItem;
import com.rapidminer.operator.learner.associations.FrequentItemSet;
import com.rapidminer.operator.learner.associations.Item;

/**
 * This class models a database which is supposed to consist of transactions,
 * that are sets of items.
 * 
 * @author Siby
 */
public class Database implements Iterable<Set<Item>> {

	private final Map<Attribute, Item> itemMap;
	private final Map<Set<Item>, Integer> transactionFreqMap;
	private int numberOfStoredItems;
	private final int numberOfTransactions;

	/**
	 * Constructs a database.
	 * 
	 * @param exampleSet
	 *            an </code>exampleSet
	 *            <code> of boolean attributes, which builds the database
	 */
	public Database(ExampleSet exampleSet) {
		itemMap = new HashMap<Attribute, Item>();
		transactionFreqMap = new HashMap<Set<Item>, Integer>();
		numberOfStoredItems = 0;
		numberOfTransactions = exampleSet.size();
		for (Attribute attribute : exampleSet.getAttributes()) {
			itemMap.put(attribute, new BooleanAttributeItem(attribute));
		}
		for (Example example : exampleSet) {
			for (Attribute attribute : exampleSet.getAttributes()) {
				// if item is bought
				if (example.getValue(attribute) == attribute.getMapping()
						.getPositiveIndex()) {
					itemMap.get(attribute).increaseFrequency();
					numberOfStoredItems++;
				}
			}
		}
		for (Example example : exampleSet) {
			Set<Item> transaction = new HashSet<Item>();
			for (Attribute attribute : exampleSet.getAttributes()) {
				// if item is bought
				if (example.getValue(attribute) == attribute.getMapping()
						.getPositiveIndex()) {
					transaction.add(itemMap.get(attribute));
				}
			}
			if (!transaction.isEmpty()) {
				if (transactionFreqMap.get(transaction) == null)
					transactionFreqMap.put(transaction, 1);
				else
					transactionFreqMap.put(transaction,
							transactionFreqMap.get(transaction) + 1);
			}

		}
	}

	/**
	 * Returns the number of stored items in this database. This number is equal
	 * to the sum of the support of all singleton itemsets.
	 * 
	 * @return the number of stored items in this database
	 */
	public int getNumberOfStoredItems() {
		return numberOfStoredItems;
	}

	/**
	 * Returns the contained items.
	 * 
	 * @return a set of the contained items
	 */
	public Set<Item> getItems() {
		return new HashSet<Item>(itemMap.values());
	}

	/**
	 * Hilfsmethode um dem dummen Fehler zu umgehen..
	 * 
	 * @param itemSet
	 * @return
	 */
	public FrequentItemSet getThisItems(FrequentItemSet freqItemSet) {
		ArrayList<Item> thisItemSet = new ArrayList<Item>();
		for (Item item : itemMap.values()) {
			if (freqItemSet.getItems().contains(item))
				thisItemSet.add(item);
		}
		return new FrequentItemSet(thisItemSet, freqItemSet.getFrequency());
	}

	@Override
	public Iterator<Set<Item>> iterator() {
		return transactionFreqMap.keySet().iterator();
	}

	/**
	 * Returns the number of transactions inside this database.
	 * 
	 * @return the number of transactions inside this database
	 */
	public int getNumberOfTransactions() {
		return numberOfTransactions;
	}

	/**
	 * Returns how often a specialized transaction is contained inside the
	 * database.
	 * 
	 * @param transaction
	 *            the transaction whose number of occurrences is determined
	 * @return the number of occurrences of the transaction in the database
	 */
	public int getFrequency(Set<Item> transaction) {
		if (transactionFreqMap.get(transaction) != null)
			return transactionFreqMap.get(transaction);
		return -1;
	}

	@Override
	public String toString() {
		String s = "";
		for (Set<Item> transaction : transactionFreqMap.keySet()) {
			s += transaction.toString() + ": "
					+ transactionFreqMap.get(transaction) + "\n";
		}
		return s;
	}

}
